/*Data Access Library. v. 1.1.
 *Datastore class.
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data;
import java.util.*;
import java.sql.*;

/**<p>Datastore JDBC objects represent datastore connections. 
 *This class manages these connections by providing client connection to the 
 *database, disconnection from the database, and the ability to commit and
 *roll back database transactions. </p>
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
*/
public class DatastoreJDBC implements java.io.Serializable , OnThreadActionExecutor{
    private String _drvName;
    private Properties _connProp;
    private Connection _conn;
    private static DatastoreJDBC _appDatastore;
    private OnThreadActionDispatcher _disp;
    private boolean _isBusy;
    private boolean _isAsync;
    private String _url;
    private Statement _stmt;

    public static final int TRANSACTION_NONE = Connection.TRANSACTION_NONE;
    public static final int TRANSACTION_READ_UNCOMMITTED =Connection.TRANSACTION_READ_UNCOMMITTED;
    public static final int TRANSACTION_READ_COMMITTED = Connection.TRANSACTION_READ_COMMITTED;
    public static final int TRANSACTION_REPEATABLE_READ = Connection.TRANSACTION_REPEATABLE_READ;
    public static final int TRANSACTION_SERIALIZABLE = Connection.TRANSACTION_SERIALIZABLE;
    public static final int TRANSACTION_NOT_SET = -1;

    protected transient ConnectionSupport _connectSupport;

    /**This is the default constructor.
     */
    public DatastoreJDBC() {
	this(null,null);
    }

    /**This constructor allows you to specify a URL that points to the database
     *you want to connect to.
     *@param url - URL of the Database.
     */
    public DatastoreJDBC(String url) {
	this(url,null);
    }

    /**This constructor allows you to specify a URL that points to the datastore
     *you want to connect to, and the properties information
     *(user ID and password) for the datastore connection.
     *@param url - URL of the Database.
     *@param prop - Properties for the database connection.
     */
    public DatastoreJDBC(String url,Properties prop) {
	_url = url;
	_connProp = prop;
	_isBusy = _isAsync = false;
	_connectSupport = new ConnectionSupport(this);
	if (_appDatastore == null)
	    setAsApplicationDatastore();
    }

    public void _executeAction(String acnN, Object[] params) throws Exception {
	if (acnN.equals("cancel()")) {
	    cancel();
	} else if (acnN.equals("commit()")) {
	    commit();
	} else if (acnN.equals("connect()")) {
	    connect();
	} else if (acnN.equals("connect(String,String)")) {
	    connect((String)params[0],(String)params[1]);
	} else if (acnN.equals("disconnect()")) {
	    disconnect();
	} else if (acnN.equals("executeSQL(String)")) {
	    executeSQL((String)params[0]);
	} else if (acnN.equals("rollback()")) {
	    rollback();
	}
    }

    public void _handleException(Exception ex) {
	ex.printStackTrace(System.err);
    }

    /**Use this method to register a listener for connection events.
     *@param l a ConnectionListener to listen for connection events.
     */
    public void addConnectionListener(ConnectionListener l) {
	_connectSupport.addConnectionListener(l);
    }


    /**This method cancels the method (for example, a database query) that
     *is currently running.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void cancel() throws DAException {
	if ( !_onBackground() && isAsynchronous() && isBusy() && _stmt != null) {
	    try {
		_stmt.cancel();
		_connectSupport.fireCancelled();
	    } catch (SQLException ex) {
		throw new DAException(DAResource.CANCEL_FAILED,ex);
	    } finally {
		_setBusy(false);
	    }
	}
    }

    /**This method commits the current set of transations to the database.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void commit() throws DAException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {};
	    _putOnBackgroundThread("commit()",params);
	    return;
	}
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    _conn.commit();
	    _connectSupport.fireCommitted();
	} catch(SQLException ex) {
	    throw new DAException(DAResource.COMMIT_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
    }

    /**This method establishes a client connection to the current datastore
     *object using default settings for the object's attributes.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     *@exception java.lang.ClassNotFoundException - if fail register driver for default URL.
     */
    public void connect() throws DAException,ClassNotFoundException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {};
	    _putOnBackgroundThread("connect()",params);
	    return;
	}
	try {
	    if (isConnected())
		disconnect();
	    _loadDriver();
	    _conn= DriverManager.getConnection(_url,_connProp);
	    _connectSupport.fireConnected("connect()");
	} catch (SQLException ex) {
	    throw new DAException(DAResource.CONNECT_FAILED,ex);
	}  finally {
	    _setBusy(false);
	}
    }

    /**This method establishes a client connection to the current datastore
     *object using the passed-in user name and password.
     *@param usr - user's name known to the Database server.
     *@param passwd - user's password.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     *@exception java.lang.ClassNotFoundException - if fail register driver for default URL.
     */
    public void connect(String usr, String passwd) throws DAException,ClassNotFoundException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {usr,passwd};
	    _putOnBackgroundThread("connect(String,String)",params);
	    return;
	}
	try {
	    if (isConnected())
		disconnect();
	    _loadDriver();
	    _conn= DriverManager.getConnection(_url,usr,passwd);
	    _connectSupport.fireConnected("connect(String,String)");
	} catch (SQLException ex) {
	    throw new DAException(DAResource.CONNECT_FAILED,ex);
	}  finally {
	    _setBusy(false);
	}
    }

    /**This method closes the connection to the datastore.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void disconnect() throws DAException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {};
	    _putOnBackgroundThread("disconnect()",params);
	    return;
	}
	try {
	    if (isConnected()) {
		_conn.close();
		_conn = null;
	    }
	    _connectSupport.fireDisconnected();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.DISCONNECT_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
    }

    /**This method executes the passed-in SQL statement in the current datastore.
     *@param qry - SQL statement.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void executeSQL(String qry) throws DAException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {qry};
	    _putOnBackgroundThread("executeSQL(String)",params);
	    return;
	}
	try {
	    if (!isConnected())
		throw new DAException(DAResource.NO_CONNECT_EXIST);
	    _stmt = _conn.createStatement();
	    _stmt.execute(qry);
	    _connectSupport.fireExecutedSQL();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.EXECUTESQL_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
    }

    /**This override of the Object finalize method checks to see if the
     *connection object is closed, and if it is not, closes it.
     */
    public void finalize() {
	try {
	    if (isConnected() && !_conn.isClosed())
		_conn.close();
	} catch(SQLException ex) {}
    }

    /**This method gets the datastore object that has been set as default for the
     *application.
     */
    public static DatastoreJDBC getApplicationDatastore() {
	return _appDatastore;
    }

    /**This method returns a <samp>java.sql.Connection</samp> object that
     *represents the client connection to the current datastore object.
     */
    public Connection getConnection() {
	return _conn;
    }

    /**Sets connection for this datastore object.
     *@param con - established client connection to the Database.
     */
    public void setConnection(Connection con) {
	try {
	    if (isConnected())
		_conn.close();
	} catch (SQLException ex) {}
	_conn = con;
    }

    /**This method gets the JDBC driver setting of the current datastore.
     */
    public String getDriver() {
	return _drvName;
    }

    /**This method gets the properties of the current datastore.
     */
    public Properties getProperties() {
	return _connProp;
    }

    /**This method gets the current connection object's transaction isolation
     * level.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public int getTransactionIsolation() throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    return _conn.getTransactionIsolation();
	} catch(SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"getTransactionIsolation()",ex);
	}
    }

    /**This method gets the URL used by the current connection.
     */
    public String getURL() {
	return _url;
    }

    /**This method tells you whether methods are set to run on a background thread
     * or not.
     */
    public boolean isAsynchronous() {
	return _isAsync;
    }

    /**This method queries the current connection object's auto-commit setting.
     *@return <sample>true</sample> if this connection is in autocommit mode.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public boolean isAutoCommit() throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    return _conn.getAutoCommit();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"isAutoCommit()",ex); 
	}
    }

    /**This method tells you whether the object is running a method in the background.
     */
    public boolean isBusy() {
	return _isBusy;
    }

    /**This method queries whether there is a connection established to the database.
     */
    public boolean isConnected() {
	return _conn != null;
    }

    /**This method returns a boolean indicating the read/write status of the
     *datastore.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public boolean isReadOnly() throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    return _conn.isReadOnly();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"isReadOnly()",ex); 
	}
    }

    /**Use this method to remove the listener on the connection object.
     *@param l - ConnectionListener object to remove.
     */
    public void removeConnectionListener(ConnectionListener l) {
	_connectSupport.removeConnectionListener(l);
    }

    //    public void removePropertyChangeListener(java.beans.PropertyChangeListener);
    /**This method rolls back all transactions since the last successful commit.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void rollback() throws DAException {
	if ( !_onBackground() && isAsynchronous() ) {
	    Object params[] = {};
	    _putOnBackgroundThread("rollback()",params);
	    return;
	}
	try {
	    if (!isConnected())
		throw new DAException(DAResource.NO_CONNECT_EXIST);
	    _conn.rollback();
	    _connectSupport.fireRolledback();
	} catch (SQLException ex) {
	    throw new DAException(DAResource.ROLLBACK_FAILED,ex);
	} finally {
	    _setBusy(false);
	}
    }

    /**This method sets the application's default datastore to the passed-in
     *datastore.
     */
    public static void setApplicationDatastore(DatastoreJDBC ds) {
	_appDatastore = ds;
    }

    /**Use this method to set the application's default datastore to the current
     *DatastoreJDBC object.
     */
    public void setAsApplicationDatastore() {
	setApplicationDatastore(this);
    }

    /**This method lets you set whether the database methods are run on a
     *background thread.
     */
    public void setAsynchronous(boolean isAs) {
	_isAsync = isAs;
    }

    /**This method lets you set the current connection object so that it
     *automatically commits all transactions.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void setAutoCommit(boolean flag) throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    _conn.setAutoCommit(flag);
	} catch (SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"setAutoCommit(boolean)",ex); 
	}
    }

    /**This method sets the JDBC driver for the database to which you want to
     *connect.
     */
    public void setDriver(String drvName) {
	_drvName = drvName;
    }

    /**This method uses the passed-in Properties object to set the user ID and
     * password properties of the current datastore.
     */
    public void setProperties(Properties prop) {
	_connProp = prop;
    }

    /**This method allows you to change the read/write access for the current
     *datastore connection.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void setReadOnly(boolean flag) throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    _conn.setReadOnly(flag);
	} catch (SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"setReadOnly(boolean)",ex); 
	}
    }

    /**This method sets the current connection object's isolation level to the
     * level you pass in.
     *@exception JProjects.eab.data.DAException - if data access error occur.
     */
    public void setTransactionIsolation(int level) throws DAException {
	if (!isConnected())
	    throw new DAException(DAResource.NO_CONNECT_EXIST);
	try {
	    _conn.setTransactionIsolation(level);
	} catch(SQLException ex) {
	    throw new DAException(DAResource.METHOD_FAILED,"setTransactionIsolation(int)",ex);
	}
    }

    /**This method sets the URL for the database to which you want to connect.
     */
    public void setURL(String url) {
	_url = url;
    }

    /**This method outputs a string indicating URL for the current database.
     */
    public String toString() {
	return "DatastoreJDBC: url= "+((_url!=null)?_url:"") + "\ndriver= " + ((_drvName!=null)?_drvName:"") + "\nisConnected = " + ((isConnected())?"true":"false");
    }

    protected void _loadDriver() throws ClassNotFoundException {
    //,InstantiationException,IllegalAccessException {
	Class.forName(_drvName);
    }

    protected boolean _onBackground() {
	if ( _disp != null )
	    return _disp.onBackground();
	return false;
    }

    protected void _putOnBackgroundThread(String acnN, Object[] params) {
	_setBusy(true);
	_disp = new OnThreadActionDispatcher(this,acnN,params);
    }

    protected void _setBusy(boolean flag) {
	_isBusy = flag;
    }


}
