package cern.lhcias.csgui.Drivers;

import cern.lhcias.csgui.interfaces.*;
import cern.lhcias.csgui.Events.*;
import cern.lhcias.csgui.WinMgr.*;
import cern.lhcias.csgui.Utils.*;
import java.lang.*;
import java.net.*;
import java.io.*;
import java.util.*;

/** This class is a socket based network driver. It implements
*<br>   the interface DataServer. It works either on a pooling base or
*<br>   in an event driven way. By default it works on an event driven way.
*<br>   The protocol used is the following:
*<br>   1/To get the values of tags, the following command is sent
*<br>        GET tagname1,tagname2,...\r\n
*<br>     The driver will then wait for the value. The returned string is 
*<br>     assumed to have the following format:
*<br>        tagname1=value1,tagname2=value2...
*<br>   2/To subscribe to some tags in an event driven way, the following
*<br>     command is sent
*<br>        SUB tagname1,tagname2,...\r\n
*<br>     It then wait for any changes on the data. The server has the 
*<br>     responsibilty to send the changes.
*<br>     Whenever the subscription tags' list has changed, a new list is sent which
*<br>     replaces the previous one.
*<br>   3/To end the subscritpion mechanism, the following command is sent:
*<br>        UNS 
*<br>   4/To set tags' values, the following command is sent
*<br>       SET tagname1=value1,tagname2=value2...\r\n
*<br>
*<br>   It is assumed that the server can process in parallel a subscription mechanism
*<br>   and atomic commands like GET or SET.
*<br>   Changing one of the property while a connection is established
*<br>   doesn't automatically close the connection. Thus it is the responsibility
*<br>   of the calling class to close and reopen a connection if needed.
*<br>   
* @see cern.lhcias.csgui.interfaces.DataServer
*<br>
*/
/* 
*<br>   user and password is not yet implemented.
*<br>   IF LIST OF TAGS HAS CHANGED, A NEW SUBSCRIPTION MUST BE SENT
*/
public class SocketDriver implements Runnable,DataServer
{
  public final static String driverType = "Socket";
  public final static String ONCHANGE = "OnChange";
  public final static String POLLING  = "Polling";
  private final static int ONCHANGETYPE = 0;
  private final static int POLLINGTYPE = 1;
  private final static int CHECKDELAY = 4000;
  
  private   int type = ONCHANGETYPE;
  protected Socket serverConn;
  String    host;
  int       port=0;
  long      poolingInterval = -1;
  boolean   mask = false;
  String    user;
  String    password;
  private   Thread dataThread;
  private   TagArrays c;
  private   String processName = "";
  private   boolean tagsSubscription = true;
  boolean   activeState = false;
  Vector    names;
  Thread    checkThread;
  checkSubmit Check;

  public void    addAlarmListener(AlarmListener listener) {}
  public void    removeAlarmListener(AlarmListener listener) {}
  
  //---------------- Constructors -----------------------------
  public SocketDriver() {
  }
  
  public SocketDriver(String Host, int Port) {
        host = Host;
        port = Port;
  }

  public SocketDriver(String Host, String portString){
        host = Host;
        port = 3000;
        try {
          port = Integer.parseInt(portString);
        }
        catch (NumberFormatException e) {}
  }

  //--------------- Properties setting ------------------------
  public void	setTagArrays(TagArrays tc) {
		c = tc;
  }
	
  public void     setReadHost(String Host) {
        host = Host;
  }
  
  public String   getReadHost() {
        return host;
  }
  public void     setReadPort(int communicationReadPort) {
        port = communicationReadPort;
  }

  public int      getReadPort() {
        return port;
  }

  public void     setWriteHost(String host_url) {}
  public String   getWriteHost() { return host; }
  public void     setWritePort(int communicationWritePort) {}
  public int      getWritePort() { return port; }

  // for pooling, if necessary
  // if pooling_interval == -1 or hasn't been set, then no pooling
  public void     setPoolingInterval(long Interval) {
    poolingInterval = Interval;
  }

  public long     getPoolingInterval() {
    return poolingInterval;
  }

  /** @param communication_type ONCHANGE or POOLING
  */
  public void setType(String Type) {
    if (Type.equalsIgnoreCase(ONCHANGE))
        type = ONCHANGETYPE;
    if (Type.equalsIgnoreCase(POLLING))
    	type = POLLINGTYPE;
  }
	
  public String getType() {
    if (type == ONCHANGETYPE) return ONCHANGE;
	else return POLLING;
  }
	
  /** if Mask is on, then the modifications to the Tags won't be sent to TagArrays
  */
  public void     setMask(boolean Mask) {
    mask = Mask;
  }

  public boolean  getMask() { return mask; }
  public void     setUser(String User) {
    user = User;
  }

  public void     setPassword(String Password) {
    password = Password;
  }

  public String   getUser() {return user;}
  public String   getPassword() {return password;}

  public void     setParameter(String parameter, String value) {}
  public String   getParameter(String parameter) { return "";}

  public boolean  isActive() {
        if (serverConn != null)
            return true;
        else
            return false;
  }

  public void     setTagsSubscription(boolean subMode) {
        tagsSubscription = subMode;
  }
  public boolean  getTagsSubscription() {
        return tagsSubscription;
  }
  public void     setProcessName(String processname) {
        processName = processname;
  }
  public String   getProcessName(){
        return processName;
  }

  public boolean doYouImplement(String type) {
    	if (type.equalsIgnoreCase(driverType)) return(true);
        return false;
  }

  public Vector getPossibleProtocols() {
        Vector types = new Vector();
        types.addElement(driverType);
        return(types);
  }

  public String getDriverProtocol(){
        return(driverType);
  }

  /** Config is in the following format
  *   ProcessName;Protocol;Type;ReadHost;ReadPort;WriteHost;WritePort;
  *   PoolFrequency;subscriptionMode[;param1=val1[;...]]
  */
  public void     setConfig(String config) {
        config = config.trim();
	    Boolean flag;
	    try {
			setProcessName(config.substring(0,config.indexOf(';')));
			config = config.substring(config.indexOf(';') + 1);
            // protocol can't be changed. It is Socket
			config = config.substring(config.indexOf(';') + 1);
			setType(config.substring(0,config.indexOf(';')));
			config = config.substring(config.indexOf(';') + 1);
			setReadHost(config.substring(0,config.indexOf(';')));
			config = config.substring(config.indexOf(';') + 1);
			setReadPort(new Integer(config.substring(0,config.indexOf(';'))).intValue());
			config = config.substring(config.indexOf(';') + 1);
			setWriteHost(config.substring(0,config.indexOf(';')));
			config = config.substring(config.indexOf(';') + 1);
			setWritePort(new Integer(config.substring(0,config.indexOf(';'))).intValue());
			config = config.substring(config.indexOf(';') + 1);
			setPoolingInterval(new Double(config.substring(0,config.indexOf(';'))).longValue());
			config = config.substring(config.indexOf(';') + 1);
			setTagsSubscription(new Boolean(config.substring(0,config.indexOf(';'))).booleanValue());
			config = config.substring(config.indexOf(';') + 1);
			Vector v = MyUtils.getPropertiesinConfig(config,';');
			for (int i=0; i<v.size()-1;i+=2) {
	            setParameter((String) v.elementAt(i), (String) v.elementAt(i+1));			    
			}
		} catch(Exception e) {}
  }
	
  /** The returned string is in the following format
  *   ProcessName;Protocol;Type;ReadHost;ReadPort;WriteHost;WritePort;
  *   PoolFrequency;subscriptionMode[;param1=val1[;...]]
  */
  public String   toString(){
        String s = processName+";"+getDriverProtocol()+";"+getType()+";"+getReadHost()
        +";"+getReadPort()+";"+getWriteHost()+";"+getWritePort()+";"+getPoolingInterval()
        +";"+getTagsSubscription();
        return s;
  }


  //------------------------- Open - Close ------------------------------------------
  /** If the connexion is not already opened, then it will be opened.
  * A thread will then be created either on POLLING or ONCHANGE mode, even if the
  * mask is on. However no change will be sent to TagArrays if mask is on.
  */
  public void enable() {
        if (host != null && port > 0) {
            if (serverConn == null) setup(host,port);
    	    dataThread = new Thread(this);
			dataThread.start();
		}
  }

  public void open() {
        if (host != null && port > 0)
           setup(host,port);
  }

  void stopThread() {
        if (dataThread != null) {
           if (dataThread.isAlive()) {
            dataThread.stop();
            dataThread = null;
           }
        }
        if (checkThread != null) 
            if (checkThread.isAlive())
                checkThread.stop();
  }
  
  /** Stop an eventual running thread and open the
  * socket channel.
  */
  public void setup(String host, int port){
        stopThread();
        try {
          serverConn = null;
          serverConn = new Socket(host, port);
        }
        catch (UnknownHostException e) {
          System.out.println("Bad host name given.");
        }
        catch (IOException e) {
          System.out.println("SocketDriver: " + e);
        }
  }

  public void disable() {
        close();
  }

  public void closeConnection() {
        close();
  }

  /** Stop an eventual running thread and close the
  * socket channel.
  */
  public void close() {
        if (type == ONCHANGETYPE && dataThread != null)
           if (dataThread.isAlive())
               sendCommandNoReturn("UNS");
        try {
            if (serverConn != null)
                serverConn.close();
        }
        catch (IOException e) {
          System.out.println("SocketDriver: " + e);
        }
        serverConn = null;
        stopThread();
  }

 private void closeFromThread() {
        if (type == ONCHANGETYPE && dataThread != null)
           if (dataThread.isAlive())
               sendCommandNoReturn("UNS");
        try {
            if (serverConn != null)
                serverConn.close();
        }
        catch (IOException e) {
          System.out.println("SocketDriver: " + e);
        }
        serverConn = null;
        if (checkThread != null) 
            if (checkThread.isAlive())
                checkThread.stop();
 }
 
 //-------------------  Sending and receiving Commands --------------------
 /** sends the command cmd to the server through the socket channel and waits
 *  for the returned string. This string is returned back.
 *  If mask is on, nothing happens and an empty String is returned.
 *  If a thread is already listening to messages from the remote server (ONCHANGE mode)
 *  then the method is not waiting for the answer ("" is returned). The answer is
 *  assumed to be received by the listening thread.
 */
 public String sendCommand(String cmd) {
    if (mask) return "";
    if (serverConn == null) open();
    if (serverConn == null) return "";
        
    try {
      DataOutputStream dout =
        new DataOutputStream(serverConn.getOutputStream());

      dout.writeBytes(cmd+"\r\n");
    
      if (!dataThread.isAlive() || type != ONCHANGETYPE) {
          BufferedReader d =
            new BufferedReader(new InputStreamReader(serverConn.getInputStream()));
          return d.readLine();
      }
    }
    catch (IOException e) {
      System.out.println("SocketDriver: " + e);
    }
    return "";
  }

 /** sends the command cmd to the server through the socket channel and returns
 *  immediately.
 *  If mask is on, nothing happens.
 */
  public void sendCommandNoReturn(String cmd) {
    if (!mask) {
        if (serverConn == null) open();
        if (serverConn == null) return;
        try {
          DataOutputStream dout =
            new DataOutputStream(serverConn.getOutputStream());
          dout.writeBytes(cmd+"\r\n");
        }
        catch (IOException e) {
          System.out.println("SocketDriver: " + e);
        }
    }
  }

 /** sends the command "SET name=stringValue" to the server 
 *  through the socket channel and returns immediately.
 *  If mask is on, nothing happens.
 */
 public void write(String name, String stringValue) {
    if (!mask)
       sendCommandNoReturn("SET "+name+"="+stringValue);
 }

 /** get through the network the value of the tag named name and returns it. 
 *  If mask is on, nothing happens and an empty String is returned.
 *  If a thread is already listening to messages from the remote server (ONCHANGE mode)
 *  then the method is not waiting for the answer ("" is returned). The answer is
 *  assumed to be received by the listening thread.
 */
 public String read(String name) {
    if (mask) return "";
    String result = sendCommand("GET "+name);
    if (result.length() < 1) return "";
    if (result == null) {
        System.out.println("SocketDriver : null returned string ");
        return "";
    }
    int pos = result.indexOf("=");
    if (pos < 1){
        System.out.println("SocketDriver : bad returned string :"+result);
        return "";
    }
    String tagName = result.substring(0,pos);
    if (!tagName.equals(name)) {
        System.out.println("SocketDriver : Inconsistancy: asked for "+name+"an got:"+result);
        return "";
    }
    return result.substring(pos+1);
 }

 /** get through the network the values of the tags listed in Vector names.
 *  The values are sent to TagArrays.
 *  If mask is on, nothing happens and false is returned.
 */
 public boolean	read(Vector names) {
    if (mask) return false;
	if (!c.beforeRefreshData()) return false;
    String get_data_string = "GET ";
    for (int i=0;i<names.size();i++) {
        get_data_string += (String) names.elementAt(i);
		if (i < (names.size()-1))
			get_data_string += ",";
	}
	String st = sendCommand(get_data_string);
    if (!dataThread.isAlive() || type != ONCHANGETYPE) {
    	update(st);
    	c.afterRefreshData();
    }
    return true;
 }

 /** get through the network the values of the tags listed in TagArrays
 *  which are linked to this DataServer.
 *  The values are sent to TagArrays.
 *  If mask is on, nothing happens.
 */
 public void read() {
    if (!mask) {
    	Vector names = c.getRemoteTagNames(this);
    	if (names != null)
    	    if (names.size() > 0) {
              String get_data_string = "GET ";
		      if (!c.beforeRefreshData()) return;
              for (int i=0;i<names.size();i++) {
                get_data_string += (String) names.elementAt(i);
		        if (i < (names.size()-1))
			        get_data_string += ",";
	          }
              update(sendCommand(get_data_string));
		      c.afterRefreshData();
	        }
	}
 }

 /** Sends a subscription request to the remote server for all the Tags
 *  listed in TagArrays which are linked to this DataServer.
 *  If mask is on, nothing happens.
 */
 public void sendSubscribeCommand() {
    if (!mask) {
        String get_data_string = "SUB ";
        
    	names = c.getRemoteTagNames(this);
        for (int i=0;i<names.size();i++) {
            get_data_string += (String) names.elementAt(i);
		    if (i < (names.size()-1))
			    get_data_string += ",";
	    }
      sendCommandNoReturn(get_data_string);
    }
 }

 /** Sends the unsubscribe commad to the remote server.
 *  If mask is on, nothing happens.
 */
 void sendUnSubscribeCommand() {
    if (!mask) {
      sendCommandNoReturn("UNS");
      if (type == ONCHANGETYPE && dataThread != null) {
        dataThread.stop();
        if (checkThread != null) 
            if (checkThread.isAlive())
                checkThread.stop();
      }
    }
 }
 
 public boolean newTags() {
    Vector newNames = c.getRemoteTagNames(this);
    if (newNames.size() != names.size()) return true;
    for (int i=0; i<newNames.size(); i++)
        if (names.indexOf(newNames.elementAt(i))<0)
            return true;
    return false;
 }

 //----------------------------------------------------------------------------
 /** Given a string in the format tag1=val1,tag2=val2,... it will update TagArrays
 */
 void update(String valueString) {
    if (valueString == null) {
        System.out.println("SocketDriver : null returned string ");
        return;
    }
    
    Vector valueVector = MyUtils.getPropertiesinConfig(valueString,',');
    for (int i=0; i< valueVector.size()-1;i+=2) {
	    c.updateValue((String) valueVector.elementAt(i),
	                  (String) valueVector.elementAt(i+1));
    }
 }

  //-------------------- managing the thread ----------------------------------
  public void run() {
        switch (type) {
        case ONCHANGETYPE: 
            try {
                sendSubscribeCommand();
                Check = new checkSubmit();
                listen();
            } catch (Exception e) {}
            return;
        case POLLINGTYPE:
    		while (serverConn != null) {
	    	    read();
		        try {
			        Thread.sleep(poolingInterval);
       			} catch (Exception e) {
        			return;
	        	}
		    }
		}
  }

  public void listen() {
    try {
        BufferedReader d =
            new BufferedReader(new InputStreamReader(serverConn.getInputStream()));
		while (serverConn != null) {
		  String st = d.readLine();
		  // System.out.println(st);
          if (!mask) { 
            update(st);
            c.afterRefreshData();
          }
		}
	}catch (IOException e) {
      System.out.println("SimpleClient: " + e);
	  closeFromThread();
    }
    if (checkThread != null) 
        if (checkThread.isAlive())
            checkThread.stop();
  }

  public synchronized void finalize() {
    System.out.println("SocketDriver: Closing down connection...");
    try { serverConn.close(); }
    catch (IOException e) {
      System.out.println("SocketDriver: " + e);
    }
  }
  
  //----------------------------------------------------------
  class checkSubmit implements Runnable {
    public checkSubmit() {
       	checkThread = new Thread(this);
       	checkThread.start();
    }

    public void run() {
      while (true) {
        if (newTags()) {
            sendSubscribeCommand();
        }
        try {
		    Thread.sleep(CHECKDELAY);
       	} catch (Exception e) {
        	return;
	    }
	  }
    }
  }
}