/**	
	Company:		Shout Interactive
	Project:		Shout3D Core Implementation
	Class:			LightTestPanel
	Date:			April 26, 1999
	Description:	Class for panel that tests light scoping
	(C) Copyright Shout Interactive, Inc. - 1997/1998/1999 - All rights reserved
 */

package applets;

import java.applet.*;
import java.awt.*;
import java.awt.image.*;
import java.io.*;
import java.util.Date;
import java.net.URL;

import shout3d.*;
import shout3d.core.*;

/**
 * LightTestPanel
 * 
 * Shows how a single Light node can be used to light
 * one of several objects, or the entire scene.
 * 
 * All this is done be setting the 'affectedGroups' field of the
 * light.
 * 
 * Clicking or dragging on any of the three colored cubes makes 
 * the light affect that cube.
 * Clicking on the bottom white bar lights everything.
 * 
 * The applet only works if the displayed file contains a light named
 * "MyLight" and  groups named "Box1" "Box2" "Box3" and "BottomBar"
 * 
 * @author Dave Westwood
 * @author Paul Isaacs
 * @author Jim Stewartson
 */

public class LightTestPanel extends Shout3DPanel implements DeviceObserver {
	
	// The light that will affect the scene
    DirectionalLight light;
	// Names that will be used for the affectedGroups field
    String box1_names[] = {"Box1"};
	String box2_names[] = {"Box2"};
	String box3_names[] = {"Box3"};
	String root_names[] = {"#Root"};  //Predefined value for lighting the entire scene
	// For determining which object is clicked with the mouse.
	Picker myPicker;
    Node[] pathToPick;

	/**
	 * Constructor
	 */
	public LightTestPanel(Shout3DApplet applet){
		super(applet);
	}
	
	/**
	 *
	 * This method is automatically called by the parent class Shout3DPanel
	 * at the correct time during initialize().
	 * 
	 * Subclasses should implement this to perform any custom initialization tasks.
	 */	
    public void customInitialize() {
		
		// Get the light, have it shine on the first group.
        light = (DirectionalLight)this.getNodeByName("MyLight");
		if (light == null){
			throw new Shout3DException("can not find node name MyLight");
		}
		else {
			light.affectedGroups.setValue(box1_names);
		}
        
		// Allocate the picker
		myPicker = getNewPicker();
		
		//Watch for mouse events to do the picking.
		this.addDeviceObserver(this, "MouseInput", null);
		
		//Call the parent class
		super.customInitialize();
	}
	
	/**
	 * Finalize
	 */
	protected void finalize(){
		this.removeDeviceObserver(this, "MouseInput");
	}

	/** 
	 * Called when Mouse input is received.
	 * 
	 * On mouse DOWN, checks for selection of a node whose 
	 * name is one of those we're interested in.  If so,
	 * sets the light to affect it.
	 * 
	 * If the selected group is "BottomBar" then sets the 
	 * affectedGroups to be the pre-defined value #Root,
	 * which Shout3d will automatically interpret to light 
	 * the entire scene.
	 *  
	 */
	public boolean onDeviceInput(DeviceInput di, Object userData){
		//No need to check type of deviceInput, only registered for Mouse Input.
		MouseInput mi = (MouseInput) di;
		if (mi.which == MouseInput.DOWN || mi.which == MouseInput.DRAG ){
			if (myPicker == null)
				return false;
		
			// Perform a pick to find what's under the cursor,
		    //
			pathToPick = myPicker.pickClosest(mi.x,mi.y);
		    if (pathToPick!=null && pathToPick.length > 0){
		        Node lastTransform = null;
				// Find the last transform on the path.
				for (int i=pathToPick.length-1;i>-1;i--){
		            if (pathToPick[i] instanceof Transform){
		                lastTransform = pathToPick[i];
		                break;
		            }
		        }
		                
				//Depending which was picked, set the name in 
				//the light's affectedGroup field.
				//The transforms for the objects are named:
				// "box1" "box2" "box3" "BottomBar"
		        if (lastTransform!=null){
                    String name = lastTransform.getDEFName();
                    if (name!=null){
                        if (name.equals(box1_names[0]))
                            light.affectedGroups.setValue(box1_names);
                        else if (name.equals(box2_names[0]))
                            light.affectedGroups.setValue(box2_names);
                        else if (name.equals(box3_names[0]))
                            light.affectedGroups.setValue(box3_names);
						else if (name.equals("BottomBar"))
                            light.affectedGroups.setValue(root_names);
                    }
                }
				//This input was used
				return true;
            }

        }
		//Did not care about this input
		return false;
    }
    

}
